<?php

namespace App\Http\Controllers\api\sales;

use Exception;
use Carbon\Carbon;
use App\models\sales\Basic;
use App\models\company\Brunch;
use App\models\customers\Customer;
use App\Http\Controllers\Controller;
use App\Http\Requests\sales\BasicRequest;
use App\models\sales\utils\PaymentMethod;
use App\Http\Requests\sales\ReportRequest;
use App\models\company\SystemFuncionality;
use App\Http\Resources\sales\BasicResource;
use App\Http\Resources\sales\BasicCollection;
use Symfony\Component\HttpFoundation\Response;

class BasicSaleController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(ReportRequest $request, $company)
    {
        //
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show(Basic $basic)
    {
        return new BasicResource($basic);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(BasicRequest $request, Basic $basic)
    {
        abort_if(!$basic->canBeChanged(), Response::HTTP_FORBIDDEN);
        abort_if(!request()->user()->can('update', $basic), Response::HTTP_FORBIDDEN);

        $customer = null;
        $availableAmount = 0;
        if ($request->filled('customer')) {
            $customer = Customer::find($request->input('customer'));
            if ($request->filled('pay_crm')) {
                $availableAmount = $customer->availableCRMAmount();
                if ($availableAmount <= 0) {
                    abort(Response::HTTP_UNPROCESSABLE_ENTITY);
                } else {
                    $basic->paymentMethod()->associate(PaymentMethod::crmpoints());
                }
            }
        }
        $basic->validate();

        if ($request->filled('customer')) {
            $amount = $basic->totalAmount();
            if ($request->filled('pay_crm')) {
                $amount = $amount - $availableAmount;
            }
            $basic->generateCRM($customer, $amount);
        }

        return response()->json([
            'body' => new BasicResource($basic),
        ], Response::HTTP_ACCEPTED);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
    }

    /**
     * Generate report of sales according to the enabled modules
     *
     * @param  Illuminate\Http\Request  $request
     * @param  App\models\company\Brunch $company
     * @param  \Carbon\Carbon  $from
     * @param  \Carbon\Carbon  $to
     * @param  bool  $dayStep
     * @param  bool  $weekStep
     * @param  bool  $monthStep
     * @param  bool  $yearStep
     *
     *  @return \Illuminate\Http\Response
     */
    public function generateReport(ReportRequest $request, Brunch $company)
    {
        $from   = (new Carbon($request->input('from')))->startOfDay();
        $to     = (new Carbon($request->input('to')))->endOfDay();

        $defaultFrom = (new Carbon($from));
        $defaultTo = (new Carbon($to));

        $hourStep = false;
        $dayStep = false;
        $weekStep = false;
        $monthStep = false;
        $yearStep = false;


        $diff = $from->diffInDays($to);
        if ($diff  <= 1)
            $hourStep = true;
        elseif ($diff < 45)
            $dayStep = true;
        elseif ($diff < 60)
            $weekStep = true;
        elseif ($diff < 367)
            $monthStep = true;
        else
            $yearStep = true;


        $report = null;
        $data = null;


        if ($request->filled('type')) {
            if ($request->input('type') == 'index') {
                $tempFrom = (new Carbon($defaultFrom));
                $tempTo = (new Carbon($defaultTo));
                $sales = $company->basicSalesPeriod($tempFrom, $tempTo)->get();
                $data =  new BasicCollection($sales);
            } else {
                $tempFrom = (new Carbon($defaultFrom));
                $tempTo = (new Carbon($defaultTo));
                $report = $this->report($company, $tempFrom, $tempTo, $hourStep, $dayStep, $weekStep, $monthStep, $yearStep);
            }
        } else {
            $tempFrom = (new Carbon($defaultFrom));
            $tempTo = (new Carbon($defaultTo));
            $report = $this->report($company, $tempFrom, $tempTo, $hourStep, $dayStep, $weekStep, $monthStep, $yearStep);

            $tempFrom = (new Carbon($defaultFrom));
            $tempTo = (new Carbon($defaultTo));
            $sales = $company->basicSalesPeriod($tempFrom, $tempTo)->get();
            $data =  new BasicCollection($sales);
        }

        $tempFrom = (new Carbon($defaultFrom));
        $tempTo = (new Carbon($defaultTo));

        $object = [
            'total' => sumSales($company->basicSalesPeriod($tempFrom, $tempTo)->get()),
            'reference' => [
                'from'  => $defaultFrom->format('d/m/Y'),
                'to'    => $defaultTo->format('d/m/Y')
            ],
            'report' => $report,
            'data' => $data
        ];
        return response()->json($object,  Response::HTTP_ACCEPTED);
    }

    private function report(Brunch $company, Carbon $from, Carbon $to, $hourStep = false, $dayStep = false, $weekStep = false, $monthStep = false, $yearStep = false)
    {
        $from = $from->startOfDay();
        $to = $to->endOfDay();

        $defaultFrom = (new Carbon($from));
        $defaultTo = (new Carbon($to));

        $data = [];

        if ($company->hasFuncionality(SystemFuncionality::basic_sale())) {
            $tempFrom = (new Carbon($defaultFrom));
            $tempTo = (new Carbon($defaultTo));

            $currentTotal = 0;
            $values = [];

            for (;;) {
                if ($hourStep)
                    $end = ((new Carbon($tempFrom))->endOfHour());
                elseif ($dayStep)
                    $end = ((new Carbon($tempFrom))->endOfDay());
                else if ($weekStep)
                    $end =  ((new Carbon($tempFrom))->endOfWeek());

                else if ($monthStep)
                    $end =   ((new Carbon($tempFrom))->endOfMonth());
                else if ($yearStep)
                    $end =  ((new Carbon($tempFrom))->endOfYear());
                else
                    throw new Exception("Step not defined");

                $partial = sumSales($company->basicSalesPeriod($tempFrom, $end)->get());
                array_push($values, $partial);
                $currentTotal = $currentTotal + $partial;

                if ($hourStep)
                    $tempFrom = ((new Carbon($tempFrom))->addHour());
                else if ($dayStep)
                    $tempFrom = ((new Carbon($tempFrom))->addDay());
                else if ($weekStep)
                    $tempFrom =  ((new Carbon($tempFrom))->addWeek());

                else if ($monthStep)
                    $tempFrom =   ((new Carbon($tempFrom))->addMonth());
                else if ($yearStep)
                    $tempFrom =  ((new Carbon($tempFrom))->addYear());

                if ($tempFrom->greaterThan($tempTo)) {
                    break;
                }
            }

            $total = $currentTotal;

            array_push($data, [
                'name' => 'BS',
                'values' => $values,
                'total' => $currentTotal
            ]);
        }

        $tempFrom = (new Carbon($defaultFrom));
        $tempTo = (new Carbon($defaultTo));

        if ($hourStep)
            $keys = hoursBetween($tempFrom, $tempTo);
        else if ($dayStep)
            $keys = daysBetween($tempFrom, $tempTo);
        else if ($weekStep)
            $keys = weeksBetween($tempFrom, $tempTo);
        else if ($monthStep)
            $keys = monthsBetween($tempFrom, $tempTo);
        else if ($yearStep)
            $keys = yearsBetween($tempFrom, $tempTo);
        return [
            'total' => $total,
            'keys' =>  $keys,
            'values' => $data
        ];
    }
}
