<?php

namespace App\Http\Controllers\api\sales;

use Exception;
use Carbon\Carbon;
use App\models\company\Brunch;
use App\models\customers\Customer;
use App\models\sales\DeliveryNote;
use App\Http\Controllers\Controller;
use App\Http\Requests\sales\ReportRequest;
use App\models\company\SystemFuncionality;
use App\Http\Requests\sales\InvoicableRequest;
use Symfony\Component\HttpFoundation\Response;
use App\Http\Requests\sales\DeliveryNoteRequest;
use App\Http\Resources\sales\DeliveryNoteResource;
use App\Http\Resources\sales\DeliveryNoteCollection;

class DeliveryNoteController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(InvoicableRequest $request)
    {
        //     $query = DeliveryNote::where('validated_at', '<>', null);
        //     $data = $query->latest()
        //         ->paginate($request->input('take', 20));
        //     return DeliveryNoteResource::collection($data);

    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function  show(DeliveryNote $deliveryNote)
    {
        return new DeliveryNoteResource($deliveryNote);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(DeliveryNoteRequest $request, DeliveryNote $deliveryNote)
    {
        abort_if(!$deliveryNote->canBeChanged(), Response::HTTP_FORBIDDEN);
        abort_if(!request()->user()->can('update', $deliveryNote), Response::HTTP_FORBIDDEN);

        $customer = Customer::find($request->input('customer'));
        $deliveryNote->obs    = $request->input('obs');
        $deliveryNote->financial_discount    = $request->input('discount', 0);
        $deliveryNote->due_date              = $request->input('due_date') ? new Carbon($request->input('due_date')) : (new Carbon())->addMonth();
        $deliveryNote->validate($customer);
        return response()->json([
            'body' => new DeliveryNoteResource($deliveryNote),
        ], Response::HTTP_ACCEPTED);
    }

    /**
     * Generate report of sales according to the enabled modules
     *
     * @param  Illuminate\Http\Request  $request
     * @param  App\models\company\Brunch $company
     * @param  \Carbon\Carbon  $from
     * @param  \Carbon\Carbon  $to
     * @param  bool  $dayStep
     * @param  bool  $weekStep
     * @param  bool  $monthStep
     * @param  bool  $yearStep
     *
     *  @return \Illuminate\Http\Response
     */
    public function generateReport(ReportRequest $request, Brunch $company)
    {
        $from   = (new Carbon($request->input('from')))->startOfDay();
        $to     = (new Carbon($request->input('to')))->endOfDay();

        $defaultFrom = (new Carbon($from));
        $defaultTo = (new Carbon($to));

        $hourStep = false;
        $dayStep = false;
        $weekStep = false;
        $monthStep = false;
        $yearStep = false;


        $diff = $from->diffInDays($to);
        if ($diff  <= 1)
            $hourStep = true;
        elseif ($diff < 45)
            $dayStep = true;
        elseif ($diff < 60)
            $weekStep = true;
        elseif ($diff < 367)
            $monthStep = true;
        else
            $yearStep = true;


        $report = null;
        $data = null;


        if ($request->filled('type')) {
            if ($request->input('type') == 'index') {
                $tempFrom = (new Carbon($defaultFrom));
                $tempTo = (new Carbon($defaultTo));
                $sales = $company->deliveryNotesPeriod($tempFrom, $tempTo)->get();
                $data =  new DeliveryNoteCollection($sales);
            } else {
                $tempFrom = (new Carbon($defaultFrom));
                $tempTo = (new Carbon($defaultTo));
                $report = $this->report($company, $tempFrom, $tempTo, $hourStep, $dayStep, $weekStep, $monthStep, $yearStep);
            }
        } else {
            $tempFrom = (new Carbon($defaultFrom));
            $tempTo = (new Carbon($defaultTo));
            $report = $this->report($company, $tempFrom, $tempTo, $hourStep, $dayStep, $weekStep, $monthStep, $yearStep);

            $tempFrom = (new Carbon($defaultFrom));
            $tempTo = (new Carbon($defaultTo));
            $sales = $company->deliveryNotesPeriod($tempFrom, $tempTo)->get();
            $data =  new DeliveryNoteCollection($sales);
        }

        $tempFrom = (new Carbon($defaultFrom));
        $tempTo = (new Carbon($defaultTo));

        $object = [
            'total' => sumSales($company->deliveryNotesPeriod($tempFrom, $tempTo)->get()),
            'reference' => [
                'from'  => $defaultFrom->format('d/m/Y'),
                'to'    => $defaultTo->format('d/m/Y')
            ],
            'report' => $report,
            'data' => $data
        ];
        return response()->json($object,  Response::HTTP_ACCEPTED);
    }

    private function report(Brunch $company, Carbon $from, Carbon $to, $hourStep = false, $dayStep = false, $weekStep = false, $monthStep = false, $yearStep = false)
    {
        $from = $from->startOfDay();
        $to = $to->endOfDay();

        $defaultFrom = (new Carbon($from));
        $defaultTo = (new Carbon($to));

        $data = [];

        if ($company->hasFuncionality(SystemFuncionality::deliveryNote())) {
            $tempFrom = (new Carbon($defaultFrom));
            $tempTo = (new Carbon($defaultTo));

            $currentTotal = 0;
            $values = [];

            for (;;) {
                if ($hourStep)
                    $end = ((new Carbon($tempFrom))->endOfHour());
                elseif ($dayStep)
                    $end = ((new Carbon($tempFrom))->endOfDay());
                else if ($weekStep)
                    $end =  ((new Carbon($tempFrom))->endOfWeek());

                else if ($monthStep)
                    $end =   ((new Carbon($tempFrom))->endOfMonth());
                else if ($yearStep)
                    $end =  ((new Carbon($tempFrom))->endOfYear());
                else
                    throw new Exception("Step not defined");

                $partial = sumSales($company->deliveryNotesPeriod($tempFrom, $end)->get());
                array_push($values, $partial);
                $currentTotal = $currentTotal + $partial;

                if ($hourStep)
                    $tempFrom = ((new Carbon($tempFrom))->addHour());
                else if ($dayStep)
                    $tempFrom = ((new Carbon($tempFrom))->addDay());
                else if ($weekStep)
                    $tempFrom =  ((new Carbon($tempFrom))->addWeek());

                else if ($monthStep)
                    $tempFrom =   ((new Carbon($tempFrom))->addMonth());
                else if ($yearStep)
                    $tempFrom =  ((new Carbon($tempFrom))->addYear());

                if ($tempFrom->greaterThan($tempTo)) {
                    break;
                }
            }

            $total = $currentTotal;

            array_push($data, [
                'name' => 'Guias de entrega',
                'values' => $values,
                'total' => $currentTotal
            ]);
        }

        $tempFrom = (new Carbon($defaultFrom));
        $tempTo = (new Carbon($defaultTo));

        if ($hourStep)
            $keys = hoursBetween($tempFrom, $tempTo);
        else if ($dayStep)
            $keys = daysBetween($tempFrom, $tempTo);
        else if ($weekStep)
            $keys = weeksBetween($tempFrom, $tempTo);
        else if ($monthStep)
            $keys = monthsBetween($tempFrom, $tempTo);
        else if ($yearStep)
            $keys = yearsBetween($tempFrom, $tempTo);
        
        return [
            'total' => $total,
            'keys' =>  $keys,
            'values' => $data
        ];
    }
}
