<?php

namespace App\Http\Controllers\api\sales;

use Exception;
use Carbon\Carbon;
use App\models\sales\Invoice;
use App\models\company\Brunch;
use App\models\customers\Customer;
use App\Http\Controllers\Controller;
use App\Http\Requests\sales\ReportRequest;
use App\models\company\SystemFuncionality;
use App\Http\Requests\sales\InvoiceRequest;
use App\Http\Resources\sales\InvoiceResource;
use App\Http\Requests\sales\InvoicableRequest;
use Symfony\Component\HttpFoundation\Response;
use App\Http\Resources\sales\DebitNoteResource;
use App\Http\Resources\sales\InvoiceCollection;
use App\Http\Resources\sales\CreditNoteResource;
use App\models\sales\relation\ItemSalableInvoice;
use App\models\sales\relation\ItemSalableDebitNote;
use App\models\sales\relation\ItemSalableCreditNote;

class InvoiceController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(InvoicableRequest $request)
    {
        $query = Invoice::where('validated_at', '<>', null);
        $data = $query->latest()
            ->paginate($request->input('take', 20));
        return InvoiceResource::collection($data);
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function  show(Invoice $invoice)
    {
        return new InvoiceResource($invoice);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(InvoiceRequest $request, Invoice $invoice)
    {
        abort_if(!$invoice->canBeChanged(), Response::HTTP_FORBIDDEN);
        abort_if(!request()->user()->can('update', $invoice), Response::HTTP_FORBIDDEN);

        $user = request()->user();
        $brunch = $user->company;

        $last_Normal_invoice = $brunch->invoices()->where('validated_at', '!=', null)->latest('created_at')->first();
        $last_By_delivery_invoice = $brunch->invoicesbydelivery()->where('validated_at', '!=', null)->latest('created_at')->first();

        $updated_id = 0;

        if (is_object($last_By_delivery_invoice) && is_object($last_Normal_invoice)) {

            if ($last_By_delivery_invoice->counter > $last_Normal_invoice->counter ) {

                $updated_id =  $last_By_delivery_invoice->counter + 1;

            }else {
                $updated_id = $last_Normal_invoice->counter + 1;
            }
        }

        if ($last_By_delivery_invoice != null && $last_Normal_invoice == null) {

                $updated_id =  $last_By_delivery_invoice->counter + 1;
            }


            $customer = Customer::find($request->input('customer'));
            $invoice->obs    = $request->input('obs');
            $invoice->financial_discount    = $request->input('discount', 0);
            $invoice->due_date              = $request->input('due_date') ? new Carbon($request->input('due_date')) : (new Carbon())->addMonth();

            if ($updated_id == 0) {
                $invoice->code = 'F.'.$invoice->id.'.'.(new Carbon($invoice->validated_at))->format("Y");
                $invoice->counter = $invoice ->id;
             } else {
                $invoice->counter = $updated_id;
                $invoice->code = 'F.'.$updated_id.'.'.(new Carbon($invoice->validated_at))->format("Y");
             }


            $invoice->validate($customer);




        return response()->json([
            'body' => new InvoiceResource($invoice),
        ], Response::HTTP_ACCEPTED);
    }


    /**
     * Generate report of sales according to the enabled modules
     *
     * @param  Illuminate\Http\Request  $request
     * @param  App\models\company\Brunch $company
     * @param  \Carbon\Carbon  $from
     * @param  \Carbon\Carbon  $to
     * @param  bool  $dayStep
     * @param  bool  $weekStep
     * @param  bool  $monthStep
     * @param  bool  $yearStep
     *
     *  @return \Illuminate\Http\Response
     */
    public function generateReport(ReportRequest $request, Brunch $company)
    {
        $from   = (new Carbon($request->input('from')))->startOfDay();
        $to     = (new Carbon($request->input('to')))->endOfDay();

        $defaultFrom = (new Carbon($from));
        $defaultTo = (new Carbon($to));

        $hourStep = false;
        $dayStep = false;
        $weekStep = false;
        $monthStep = false;
        $yearStep = false;


        $diff = $from->diffInDays($to);
        if ($diff  <= 1)
            $hourStep = true;
        elseif ($diff < 45)
            $dayStep = true;
        elseif ($diff < 60)
            $weekStep = true;
        elseif ($diff < 367)
            $monthStep = true;
        else
            $yearStep = true;


        $report = null;
        $data = null;


        if ($request->filled('type')) {
            if ($request->input('type') == 'index') {
                $tempFrom = (new Carbon($defaultFrom));
                $tempTo = (new Carbon($defaultTo));
                $sales = $company->invoicesPeriod($tempFrom, $tempTo)->get();
                $data =  new InvoiceCollection($sales);
            } else {
                $tempFrom = (new Carbon($defaultFrom));
                $tempTo = (new Carbon($defaultTo));
                $report = $this->report($company, $tempFrom, $tempTo, $hourStep, $dayStep, $weekStep, $monthStep, $yearStep);
            }
        } else {
            $tempFrom = (new Carbon($defaultFrom));
            $tempTo = (new Carbon($defaultTo));
            $report = $this->report($company, $tempFrom, $tempTo, $hourStep, $dayStep, $weekStep, $monthStep, $yearStep);

            $tempFrom = (new Carbon($defaultFrom));
            $tempTo = (new Carbon($defaultTo));
            $sales = $company->invoicesPeriod($tempFrom, $tempTo)->get();
            $data =  new InvoiceCollection($sales);
        }

        $tempFrom = (new Carbon($defaultFrom));
        $tempTo = (new Carbon($defaultTo));

        $object = [
            'total' => sumSales($company->invoicesPeriod($tempFrom, $tempTo)->get()),
            'reference' => [
                'from'  => $defaultFrom->format('d/m/Y'),
                'to'    => $defaultTo->format('d/m/Y')
            ],
            'report' => $report,
            'data' => $data
        ];
        return response()->json($object,  Response::HTTP_ACCEPTED);
    }

    private function report(Brunch $company, Carbon $from, Carbon $to, $hourStep = false, $dayStep = false, $weekStep = false, $monthStep = false, $yearStep = false)
    {
        $from = $from->startOfDay();
        $to = $to->endOfDay();

        $defaultFrom = (new Carbon($from));
        $defaultTo = (new Carbon($to));

        $data = [];

        if ($company->hasFuncionality(SystemFuncionality::invoice())) {
            $tempFrom = (new Carbon($defaultFrom));
            $tempTo = (new Carbon($defaultTo));

            $currentTotal = 0;
            $values = [];

            for (;;) {
                if ($hourStep)
                    $end = ((new Carbon($tempFrom))->endOfHour());
                elseif ($dayStep)
                    $end = ((new Carbon($tempFrom))->endOfDay());
                else if ($weekStep)
                    $end =  ((new Carbon($tempFrom))->endOfWeek());

                else if ($monthStep)
                    $end =   ((new Carbon($tempFrom))->endOfMonth());
                else if ($yearStep)
                    $end =  ((new Carbon($tempFrom))->endOfYear());
                else
                    throw new Exception("Step not defined");

                $partial = sumSales($company->invoicesPeriod($tempFrom, $end)->get());
                array_push($values, $partial);
                $currentTotal = $currentTotal + $partial;

                if ($hourStep)
                    $tempFrom = ((new Carbon($tempFrom))->addHour());
                else if ($dayStep)
                    $tempFrom = ((new Carbon($tempFrom))->addDay());
                else if ($weekStep)
                    $tempFrom =  ((new Carbon($tempFrom))->addWeek());

                else if ($monthStep)
                    $tempFrom =   ((new Carbon($tempFrom))->addMonth());
                else if ($yearStep)
                    $tempFrom =  ((new Carbon($tempFrom))->addYear());

                if ($tempFrom->greaterThan($tempTo)) {
                    break;
                }
            }

            $total = $currentTotal;

            array_push($data, [
                'name' => 'Faturas',
                'values' => $values,
                'total' => $currentTotal
            ]);
        }

        $tempFrom = (new Carbon($defaultFrom));
        $tempTo = (new Carbon($defaultTo));

        if ($hourStep)
            $keys = hoursBetween($tempFrom, $tempTo);
        else if ($dayStep)
            $keys = daysBetween($tempFrom, $tempTo);
        else if ($weekStep)
            $keys = weeksBetween($tempFrom, $tempTo);
        else if ($monthStep)
            $keys = monthsBetween($tempFrom, $tempTo);
        else if ($yearStep)
            $keys = yearsBetween($tempFrom, $tempTo);
        return [
            'total' => $total,
            'keys' =>  $keys,
            'values' => $data
        ];
    }


    public function copyToBasicsale(Invoice $invoice)
    {
    }
    public function copyToInvoice(Invoice $invoice)
    {
    }
    public function copyToCashsale(Invoice $invoice)
    {
    }
    public function copyToProforma(Invoice $invoice)
    {
    }
    public function copyToCreditnote(Invoice $invoice)
    {
        $user = request()->user();
        $creditnote = $user->createCreditNote();
        $creditnote->financial_discount = $invoice->financial_discount;
        $creditnote->save();

        $invoice->saleItens->each(function (ItemSalableInvoice $saleItem) use ($creditnote) {

            $item = $creditnote->hasSaleItem($saleItem->saleable);
            $item =  $item ? $item : new ItemSalableCreditNote();
            $item->creditnote()->associate($creditnote);
            $item->saleable()->associate($saleItem->saleable);
            $item->sale_code    = $saleItem->sale_code;
            $item->sale_name    = $saleItem->sale_name;
            $item->quantity     = $saleItem->quantity;
            $item->unit_amount  = $saleItem->unit_amount;
            $item->discount     = $saleItem->discount;
            $item->vat_tax      = $saleItem->vat_tax;
            $item->save();
        });
        return new CreditNoteResource($creditnote);
    }

    public function copyToDebitnote(Invoice $invoice)
    {
        $user = request()->user();
        $debitnote = $user->createDebitNote();
        $debitnote->financial_discount = $invoice->financial_discount;
        $debitnote->save();

        $invoice->saleItens->each(function (ItemSalableInvoice $saleItem) use ($debitnote) {

            $item = $debitnote->hasSaleItem($saleItem->saleable);
            $item =  $item ? $item : new ItemSalableDebitNote();
            $item->debitnote()->associate($debitnote);
            $item->saleable()->associate($saleItem->saleable);
            $item->sale_code    = $saleItem->sale_code;
            $item->sale_name    = $saleItem->sale_name;
            $item->quantity     = $saleItem->quantity;
            $item->unit_amount  = $saleItem->unit_amount;
            $item->discount     = $saleItem->discount;
            $item->vat_tax      = $saleItem->vat_tax;
            $item->save();
        });
        return new DebitNoteResource($debitnote);
    }

    public function fecthInvoice($code){

        $invoice = Invoice::where('code', 'like', '%'.$code.'%')->first();
        $items = $invoice->saleItens()->get();


        return [$invoice, $items];
    }
}
