<?php

namespace App\Http\Controllers\api\sales;

use Exception;
use Carbon\Carbon;
use App\models\company\Brunch;
use App\models\sales\Proforma;
use App\models\customers\Customer;
use App\Http\Controllers\Controller;
use App\Http\Requests\sales\ReportRequest;
use App\models\company\SystemFuncionality;
use App\Http\Resources\sales\BasicResource;
use App\Http\Requests\sales\ProformaRequest;
use App\Http\Resources\sales\InvoiceResource;
use App\Http\Resources\sales\CashSaleResource;
use App\Http\Resources\sales\ProformaResource;
use Symfony\Component\HttpFoundation\Response;
use App\Http\Resources\sales\DebitNoteResource;
use App\models\sales\relation\ItemSalableBasic;
use App\Http\Requests\sales\ProformaCopyRequest;
use App\Http\Resources\sales\CreditNoteResource;
use App\Http\Resources\sales\ProformaCollection;
use App\models\sales\relation\ItemSalableInvoice;
use App\models\sales\relation\ItemSalableCashSale;
use App\models\sales\relation\ItemSalableProforma;
use App\models\sales\relation\ItemSalableDebitNote;
use App\models\sales\relation\ItemSalableCreditNote;
use App\Http\Controllers\api\sales\utils\CanCopyDocument;


class ProformaController extends Controller
{

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(ReportRequest $request, $company)
    {
        //
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show(Proforma $proforma)
    {
        return new ProformaResource($proforma);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(ProformaRequest $request, Proforma $proforma)
    {
        abort_if(!$proforma->canBeChanged(), Response::HTTP_FORBIDDEN);
        abort_if(!request()->user()->can('update', $proforma), Response::HTTP_FORBIDDEN);

        $customer = null;
        if ($request->filled('customer')) {
            $customer = Customer::find($request->input('customer'));
        }

        $proforma->financial_discount    = $request->input('discount', 0);
        $proforma->validate($customer);

        $proforma->due_date              = $request->input('due_date') ? new Carbon($request->input('due_date')) : (new Carbon())->addWeek();
        if (!$request->filled('customer')) {
            $proforma->customer_name = $request->input('customer_name');
            $proforma->customer_nuit = $request->input('customer_nuit');
            $proforma->customer_address = $request->input('customer_address');
        }
        $proforma->save();


        return response()->json([
            'body' => new ProformaResource($proforma),
        ], Response::HTTP_ACCEPTED);
    }


    use CanCopyDocument;
    public function copy(ProformaCopyRequest $request, Proforma $proforma)
    {
        $document   = null;
        $target     = null;
        $user = request()->user();


        abort_if(!$proforma->isValid(), Response::HTTP_FORBIDDEN);
        abort_if(!request()->user()->can('view', $proforma), Response::HTTP_FORBIDDEN);
        $document  = $proforma;


        $destiny = $request->input('destiny');
        if ($destiny == 'invoice') {
            abort_unless($user->can('issue-invoice'), Response::HTTP_FORBIDDEN);
            $target = $user->createInvoice();
            $result = $this->copyDocument($document, $target, ItemSalableInvoice::class);
            return response()->json(
                new InvoiceResource($result),
                Response::HTTP_CREATED
            );
        }
        if ($destiny == 'proforma') {
            abort_unless($user->can('issue-proforma'), Response::HTTP_FORBIDDEN);
            $target = $user->createProforma();
            $result = $this->copyDocument($document, $target, ItemSalableProforma::class);
            return response()->json(
                new ProformaResource($result),
                Response::HTTP_CREATED
            );
        }
        if ($destiny == 'cashsale') {
            abort_unless($user->can('issue-cashsale'), Response::HTTP_FORBIDDEN);
            $target = $user->createCashSale();
            $result = $this->copyDocument($document, $target, ItemSalableCashSale::class);
            return response()->json(
                new CashSaleResource($result),
                Response::HTTP_CREATED
            );
        }
        if ($destiny == 'basicsale') {
            abort_unless($user->can('issue-basicsale'), Response::HTTP_FORBIDDEN);
            $target = $user->createBasicSale();
            $result = $this->copyDocument($document, $target, ItemSalableBasic::class);
            return response()->json(
                new BasicResource($result),
                Response::HTTP_CREATED
            );
        }
        if ($destiny == 'debitnote') {
            abort_unless($user->can('issue-debitnote'), Response::HTTP_FORBIDDEN);
            $target = $user->createDebitNote();
            $result = $this->copyDocument($document, $target, ItemSalableDebitNote::class);
            return response()->json(
                new DebitNoteResource($result),
                Response::HTTP_CREATED
            );
        }
        if ($destiny == 'creditnote') {
            abort_unless($user->can('issue-creditnote'), Response::HTTP_FORBIDDEN);
            $target = $user->createCreditNote();
            $result = $this->copyDocument($document, $target, ItemSalableCreditNote::class);
            return response()->json(
                new CreditNoteResource($result),
                Response::HTTP_CREATED
            );
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
    }

    /**
     * Generate report of sales according to the enabled modules
     *
     * @param  Illuminate\Http\Request  $request
     * @param  App\models\company\Brunch $company
     * @param  \Carbon\Carbon  $from
     * @param  \Carbon\Carbon  $to
     * @param  bool  $dayStep
     * @param  bool  $weekStep
     * @param  bool  $monthStep
     * @param  bool  $yearStep
     *
     *  @return \Illuminate\Http\Response
     */
    public function generateReport(ReportRequest $request, Brunch $company)
    {
        $from   = (new Carbon($request->input('from')))->startOfDay();
        $to     = (new Carbon($request->input('to')))->endOfDay();

        $defaultFrom = (new Carbon($from));
        $defaultTo = (new Carbon($to));

        $hourStep = false;
        $dayStep = false;
        $weekStep = false;
        $monthStep = false;
        $yearStep = false;


        $diff = $from->diffInDays($to);
        if ($diff  <= 1)
            $hourStep = true;
        elseif ($diff < 45)
            $dayStep = true;
        elseif ($diff < 60)
            $weekStep = true;
        elseif ($diff < 367)
            $monthStep = true;
        else
            $yearStep = true;


        $report = null;
        $data = null;


        if ($request->filled('type')) {
            if ($request->input('type') == 'index') {
                $tempFrom = (new Carbon($defaultFrom));
                $tempTo = (new Carbon($defaultTo));
                $sales = $company->proformasPeriod($tempFrom, $tempTo)->get();
                $data =  new ProformaCollection($sales);
            } else {
                $tempFrom = (new Carbon($defaultFrom));
                $tempTo = (new Carbon($defaultTo));
                $report = $this->report($company, $tempFrom, $tempTo, $hourStep, $dayStep, $weekStep, $monthStep, $yearStep);
            }
        } else {
            $tempFrom = (new Carbon($defaultFrom));
            $tempTo = (new Carbon($defaultTo));
            $report = $this->report($company, $tempFrom, $tempTo, $hourStep, $dayStep, $weekStep, $monthStep, $yearStep);

            $tempFrom = (new Carbon($defaultFrom));
            $tempTo = (new Carbon($defaultTo));
            $sales = $company->proformasPeriod($tempFrom, $tempTo)->get();
            $data =  new ProformaCollection($sales);
        }

        $tempFrom = (new Carbon($defaultFrom));
        $tempTo = (new Carbon($defaultTo));

        $object = [
            'total' => sumSales($company->proformasPeriod($tempFrom, $tempTo)->get()),
            'reference' => [
                'from'  => $defaultFrom->format('d/m/Y'),
                'to'    => $defaultTo->format('d/m/Y')
            ],
            'report' => $report,
            'data' => $data
        ];
        return response()->json($object,  Response::HTTP_ACCEPTED);
    }

    private function report(Brunch $company, Carbon $from, Carbon $to, $hourStep = false, $dayStep = false, $weekStep = false, $monthStep = false, $yearStep = false)
    {
        $from = $from->startOfDay();
        $to = $to->endOfDay();

        $defaultFrom = (new Carbon($from));
        $defaultTo = (new Carbon($to));

        $data = [];

        if ($company->hasFuncionality(SystemFuncionality::proforma())) {
            $tempFrom = (new Carbon($defaultFrom));
            $tempTo = (new Carbon($defaultTo));

            $currentTotal = 0;
            $values = [];

            for (;;) {
                if ($hourStep)
                    $end = ((new Carbon($tempFrom))->endOfHour());
                elseif ($dayStep)
                    $end = ((new Carbon($tempFrom))->endOfDay());
                else if ($weekStep)
                    $end =  ((new Carbon($tempFrom))->endOfWeek());

                else if ($monthStep)
                    $end =   ((new Carbon($tempFrom))->endOfMonth());
                else if ($yearStep)
                    $end =  ((new Carbon($tempFrom))->endOfYear());
                else
                    throw new Exception("Step not defined");

                $partial = sumSales($company->proformasPeriod($tempFrom, $end)->get());
                array_push($values, $partial);
                $currentTotal = $currentTotal + $partial;

                if ($hourStep)
                    $tempFrom = ((new Carbon($tempFrom))->addHour());
                else if ($dayStep)
                    $tempFrom = ((new Carbon($tempFrom))->addDay());
                else if ($weekStep)
                    $tempFrom =  ((new Carbon($tempFrom))->addWeek());

                else if ($monthStep)
                    $tempFrom =   ((new Carbon($tempFrom))->addMonth());
                else if ($yearStep)
                    $tempFrom =  ((new Carbon($tempFrom))->addYear());

                if ($tempFrom->greaterThan($tempTo)) {
                    break;
                }
            }

            $total = $currentTotal;

            array_push($data, [
                'name' => 'Proforma',
                'values' => $values,
                'total' => $currentTotal
            ]);
        }

        $tempFrom = (new Carbon($defaultFrom));
        $tempTo = (new Carbon($defaultTo));

        if ($hourStep)
            $keys = hoursBetween($tempFrom, $tempTo);
        else if ($dayStep)
            $keys = daysBetween($tempFrom, $tempTo);
        else if ($weekStep)
            $keys = weeksBetween($tempFrom, $tempTo);
        else if ($monthStep)
            $keys = monthsBetween($tempFrom, $tempTo);
        else if ($yearStep)
            $keys = yearsBetween($tempFrom, $tempTo);
        return [
            'total' => $total,
            'keys' =>  $keys,
            'values' => $data
        ];
    }
}
