<?php

namespace App\Http\Controllers\api\sales\relation;

use App\models\sales\DeliveryNote;
use App\models\stock\Saleable;
use App\models\stock\Warehouse;
use App\Http\Controllers\Controller;
use App\Http\Resources\sales\DeliveryNoteResource;
use Symfony\Component\HttpFoundation\Response;
use App\models\sales\relation\ItemSalableDeliveryNote;
use App\Http\Requests\sales\relation\ItemSalableDeliveryNoteRequest;

class ItemSalableDeliveryNoteController extends Controller
{

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(ItemSalableDeliveryNoteRequest $request, DeliveryNote $deliveryNote)
    {
        abort_if(!$deliveryNote->canBeChanged(), Response::HTTP_FORBIDDEN);
        abort_if(!request()->user()->can('update', $deliveryNote), Response::HTTP_FORBIDDEN);

        $alerts =  [];

        $sale_code = $request->input('sale_code');
        $sale_name = $request->input('sale_name');
        $unit_amount = $request->input('unit_amount');
        $vat_tax = $request->input('vat_tax');

        $item =  new ItemSalableDeliveryNote();

        if ($request->filled('saleable')) {
            $saleable = Saleable::find($request->input('saleable'));

            $item = $deliveryNote->hasSaleItem($saleable);

            $item =  $item ? $item : new ItemSalableDeliveryNote();

            $item->saleable()->associate($saleable);

            $sale_code      = $sale_code ? $sale_code : $saleable->code;
            $sale_name      = $sale_name ? $sale_name : $saleable->name;
            $unit_amount    = $unit_amount ? $unit_amount : $saleable->price;
            $vat_tax        = $vat_tax ? $vat_tax : $saleable->vatScheme->tax;
        }
        $item->deliveryNote()->associate($deliveryNote);


        $item->sale_code = $sale_code;
        $item->sale_name = $sale_name;
        $item->quantity = $item->quantity + $request->input('quantity');
        $item->unit_amount = $unit_amount;
        $item->discount = $request->input('discount', 0);
        $item->vat_tax = $vat_tax;



        if ($request->filled('warehouse')) {
            $warehouse = Warehouse::find($request->input('warehouse'));
            abort_if(!request()->user()->can('view', $warehouse), Response::HTTP_FORBIDDEN);
            $item->warehouse()->associate($warehouse);

            $move = $warehouse->lastMove($saleable);

            if ($move) {

                if (($move->balance - $item->quantity)  < 0) {
                    array_push($alerts, 'Quantidade indisponível em '.$warehouse->name.'<br/> <br/> Disponível: '.number_format($move->balance));
                } else {

                    $item->save();
                }

            } else
                array_push($alerts, __('Não existem entradas do produto no armazém selecionado'));
        }





        return response()->json([
            'message' => 'OK',
            'alerts' => $alerts,
            'body' => new DeliveryNoteResource($deliveryNote)
        ], Response::HTTP_CREATED);
    }



    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(DeliveryNote $deliveryNote, ItemSalableDeliveryNote $item)
    {
        abort_if(!$deliveryNote->canBeChanged(), Response::HTTP_FORBIDDEN);
        abort_if(!$item->deliveryNote->is($deliveryNote), Response::HTTP_FORBIDDEN);
        abort_if(!request()->user()->can('update', $deliveryNote), Response::HTTP_FORBIDDEN);

        $item->forceDelete();
        return response()->json([
            'message' => 'OK',
        ], Response::HTTP_NO_CONTENT);
    }
}
