<?php

namespace App\Http\Controllers\api\stock;

use Carbon\Carbon;
use App\models\company\Brunch;
use App\models\stock\Saleable;
use App\models\stock\SaleUnit;
use App\Imports\SaleableImport;
use App\models\stock\VatScheme;
use App\models\stock\Warehouse;
use Illuminate\Support\Facades\URL;
use App\Http\Controllers\Controller;
use Maatwebsite\Excel\Facades\Excel;
use App\Http\Requests\SaleableReportRequest;
use App\Http\Requests\stock\SaleableRequest;
use Symfony\Component\HttpFoundation\Response;
use App\Http\Resources\stock\SaleableCollection;
use App\Http\Resources\stock\StockMoveCollection;
use App\Http\Requests\reports\GeneralSalesReportRequest;

class SaleableController extends Controller
{
    public function sales(GeneralSalesReportRequest $request)
    {
        $user = request()->user();
        $from = (new Carbon($request->input('from', (new Carbon())->startOfYear())))->startOfDay();
        $to = (new Carbon($request->input('to', now())))->endOfDay();

        $company = $user->company;
        $saleables = $user->company->saleables()
            ->whereHas('basicSales', function ($query) use ($from, $to, $company) {
                return $query
                    ->whereHas('company', function ($query) use($company){
                        return $query->where(Brunch::table() . '.id', $company->id);
                    })
                    ->where('validated_at', '>=', $from)
                    ->where('validated_at', '<=', $to)
                    ->where('anuled_at', '=', null);
            })
            ->orWhereHas('invoices', function ($query) use ($from, $to, $company) {
                return $query
                    ->whereHas('company', function ($query) use($company){
                        return $query->where(Brunch::table() . '.id', $company->id);
                    })
                    ->where('validated_at', '>=', $from)
                    ->where('validated_at', '<=', $to)
                    ->where('anuled_at', '=', null);
            })
            ->orWhereHas('debitNotes', function ($query) use ($from, $to, $company) {
                return $query
                    ->whereHas('company', function ($query) use($company){
                        return $query->where(Brunch::table() . '.id', $company->id);
                    })
                    ->where('validated_at', '>=', $from)
                    ->where('validated_at', '<=', $to)
                    ->where('anuled_at', '=', null);
            })
            ->orWhereHas('cashSales', function ($query) use ($from, $to, $company) {
                return $query
                    ->whereHas('company', function ($query) use($company){
                        return $query->where(Brunch::table() . '.id', $company->id);
                    })
                    ->where('validated_at', '>=', $from)
                    ->where('validated_at', '<=', $to)
                    ->where('anuled_at', '=', null);
            })
            ->with([
                'saleUnit', 'vatScheme'
            ])
            ->get();


        $saleables->each(function (Saleable &$item) use ($from, $to) {
            $item->sales = $item->salesOnPeriod($from, $to);
        });

        return new SaleableCollection($saleables);
    }


    public function statement(Saleable $saleable, SaleableReportRequest $request){
        $warehouse = Warehouse::find($request->input('warehouse'));
        $moves = $saleable->stockMovesByWarehouse($warehouse)->get();
        return new StockMoveCollection($moves);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(SaleableRequest $request)
    {
        if ($request->hasFile('file')) {
            return $this->import($request->file('file'));
        }

        $vatScheme = VatScheme::find($request->input('vat_scheme'));

        $saleable = new Saleable();
        $saleable->code = $request->input('code');
        $saleable->name = $request->input('name');
        $saleable->description = $request->input('description');
        $saleable->is_stockable = $request->input('stockable', false) ? true : false;

        //Tornado os produtos temporarios

        $saleable->temp = 1;    // se tem zero o produto temporário esta desabilitado
        $saleable->deleted_at = Carbon::now();

        if ($request->filled('calculate_vat_scheme')) {
            $saleable->price = $request->input('price') + ($vatScheme->multiplyFactor() * $request->input('price'));
        } else {
            $saleable->price = $request->input('price');
        }

        $saleable->saleUnit()->associate(SaleUnit::find($request->input('sale_unit')));
        $saleable->vatScheme()->associate($vatScheme);
        $saleable->company()->associate($request->user()->company);
        $saleable->save();
        $saleable->categories()->sync($request->input('categories'));

        return response()->json([
            'message' => 'OK',
            'url' => [
                'web' => URL::route('web.stock.saleable.create')
            ]
        ], Response::HTTP_CREATED);
    }


    private function import($file)
    {
        $result = Excel::toArray(new SaleableImport, $file)[0];
        $company = request()->user()->company;

        foreach (collect($result)  as $key => $row) {
            if (!$key)
                continue;

            /* Verification of name */
            if ($row[1] == null)
                return response()->json([
                    'message' => 'Failed',
                    'description' => __('Falha com nome do produto'),
                ], Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        $saleables = collect([]);

        foreach (collect($result)  as $key => $row) {
            if (!$key)
                continue;

            if ($company->saleables()->whereCode($row[0])->doesntExist()) {
                $saleable = new Saleable();
                $saleable->code = $row[0];
                $saleable->name = $row[1];
                $saleable->description = $row[2];
                $saleable->is_stockable = $row[3] ? false : true;
                $saleable->min_quantity = $row[4];
                $saleable->price = $row[5];
                $vatScheme = VatScheme::whereTax($row[7])->first() ?? VatScheme::normal();
                $saleable->vatScheme()->associate($vatScheme);
                $saleUnit = SaleUnit::find($row[6]) ?? SaleUnit::first();
                $saleable->saleUnit()->associate($saleUnit);
                $saleable->company()->associate($company);
                $saleable->save();
                $saleables->push($saleable);


                if (!is_null($row[8])) {
                    $category = $company->categories()->whereName($row[8])->first();
                    if ($category) {
                        $category->saleables()->attach($saleable);
                    }
                }
            }
        }
        return response()->json([
            new SaleableCollection($saleables)
        ], Response::HTTP_CREATED);
    }


    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(SaleableRequest $request, Saleable $saleable)
    {
        $user = request()->user();
        abort_if(!$user->can('update', $saleable), Response::HTTP_FORBIDDEN);

        $saleable->code = $request->input('code');
        $saleable->name = $request->input('name');
        $saleable->description = $request->input('description');
        $saleable->is_stockable = $request->input('stockable', false) ? true : false;
        $saleable->price = $request->input('price');
        $saleable->saleUnit()->associate(SaleUnit::find($request->input('sale_unit')));
        $saleable->vatScheme()->associate(VatScheme::find($request->input('vat_scheme')));
        $saleable->save();
        $saleable->categories()->sync($request->input('categories'));

        return response()->json([
            'message' => 'OK',
            'url' => [
                'web' => URL::route('web.stock.saleable.show', ['saleable' => $saleable->id])
            ]
        ], Response::HTTP_ACCEPTED);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Saleable $saleable)
    {
        $user = request()->user();
        abort_if(!$user->can('delete', $saleable), Response::HTTP_FORBIDDEN);

        $saleable->forceDelete();

        return response()->json([
            'message' => 'OK',
        ], Response::HTTP_NO_CONTENT);
    }
}
