<?php

namespace App\Http\Controllers\api\stock;

use Carbon\Carbon;
use Illuminate\Http\Request;
use App\models\stock\Category;
use App\models\stock\Saleable;
use App\Imports\SaleableImport;
use App\models\stock\StockMove;
use App\models\stock\Warehouse;
use App\models\provider\Purchase;
use App\models\stock\StockMoveType;
use App\Http\Controllers\Controller;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Validator;
use App\Http\Requests\stock\StockMoveRequest;
use Symfony\Component\HttpFoundation\Response;
use App\Http\Requests\stock\StockAdjustRequest;
use App\Http\Resources\stock\StockMoveCollection;
use App\Http\Requests\stock\StockMoveFilterRequest;

class StockMoveController extends Controller
{

    public function inventory(StockMoveFilterRequest $request)
    {
        $warehouse = Warehouse::find($request->input('warehouse'));
        abort_if(!request()->user()->can('view', $warehouse), Response::HTTP_FORBIDDEN);

        if ($request->filled('categories'))
            $moves = $warehouse->saleablesInventory(Category::find($request->input('categories'))->pluck('id'));
        else
            $moves = $warehouse->saleablesInventory();
        return new StockMoveCollection($moves);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'purchase' => 'nullable|integer|exists:purchases,id',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors()->toArray(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        $stockmoves = null;
        if ($request->filled('purchase')) {
            $purchase = Purchase::find($request->input('purchase'));
            abort_if(!request()->user()->can('view', $purchase), Response::HTTP_FORBIDDEN);
            $stockmoves = $purchase->stockMoves()->get();
        } else {
            $company = request()->user()->company;
            $stockmoves = $company->stockMoves()->doesntHave('purchase')->where('validated_at', null)->get();
        }
        return new StockMoveCollection($stockmoves);
    }
    public function adjust(StockAdjustRequest $request)
    {
        
        $saleable = Saleable::find($request->input('saleable'));
        $warehouse = Warehouse::find($request->input('warehouse'));
        $quantity  = $request->input('quantity');


        if (!$saleable->company->is($request->user()->company)) {
            abort(Response::HTTP_FORBIDDEN);
        }

        if (!$warehouse->company->is($request->user()->company)) {
            abort(Response::HTTP_FORBIDDEN);
        }
        $user = request()->user();

        $move = StockMove::adjust($saleable, $warehouse, $quantity, $user);
        $company = request()->user()->company;
        $stockmoves = $company->stockMoves()->doesntHave('purchase')->where('validated_at', null)->get();

        return response()->json([
            'data' => new StockMoveCollection($stockmoves)
        ], Response::HTTP_CREATED);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(StockMoveRequest $request)
    {

        if ($request->hasFile('file')) {
            return $this->import($request->file('file'));
        }

        $purchase = null;
        if ($request->filled('purchase')) {
            $purchase = Purchase::find($request->input('purchase'));
            // abort_if(!request()->user()->can('update', $purchase), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        $saleable = Saleable::find($request->input('saleable'));
        $warehouse = Warehouse::find($request->input('warehouse'));


        if (!$saleable->company->is($request->user()->company)) {
            abort(Response::HTTP_FORBIDDEN);
        }

        if (!$warehouse->company->is($request->user()->company)) {
            abort(Response::HTTP_FORBIDDEN);
        }


        $moveType = StockMoveType::entry();
        $quantity  = $request->input('quantity');
        $price  = $request->input('price');
        $user = request()->user();
        $expirity_date = $request->input('expirity_date', null) ? new Carbon($request->input('expirity_date', null)) : null;


        $move = StockMove::move($saleable, $warehouse, $moveType, $user,  $quantity, $price, $expirity_date);

        $company = request()->user()->company;
        $stockmoves = $company->stockMoves()->doesntHave('purchase')->where('validated_at', null)->get();



        if ($purchase) {
            $move->purchase()->associate($purchase);
            $move->save();
            $stockmoves = $purchase->stockMoves;
        }
        return response()->json([
            'data' => new StockMoveCollection($stockmoves)
        ], Response::HTTP_CREATED);
    }


    private function import($file)
    {
        $result = Excel::toArray(new SaleableImport, $file)[0];
        $company = request()->user()->company;

        foreach (collect($result)  as $key => $row) {
            if (!$key)
                continue;
            /* Verification of code */
            if ($row[0] == null)
                return response()->json([
                    'message' => 'Failed',
                    'description' => __('Falha com nome do produto'),
                ], Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        $stockmoves = collect([]);

        foreach (collect($result)  as $key => $row) {
            if (!$key)
                continue;

            $saleable       = $company->saleables()->whereCode($row[0])->first();
            $warehouse      = $company->warehouses()->whereName($row[4])->first();
            $quantity       = $row[1];
            $price          = $row[2];
            $expirity_date  = (!is_null($row[3])) ? new Carbon(is_null($row[3])) : null;

            if (!$saleable)
                continue;

            if (!$warehouse)
                continue;

            if (!is_numeric($price))
                continue;

            $moveType = StockMoveType::entry();
            $user = request()->user();
            $move = StockMove::move($saleable, $warehouse, $moveType, $user,  $quantity, $price, $expirity_date);
            $stockmoves = $company->stockMoves()->doesntHave('purchase')->where('validated_at', null)->get();
        }

        return response()->json([
            'message' => 'OK',
            new StockMoveCollection($stockmoves)
        ], Response::HTTP_CREATED);
    }
    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(StockMove $move)
    {
        abort_if(!request()->user()->can('delete', $move), Response::HTTP_FORBIDDEN);

        if (!$move->isValid()) {
            $move->forceDelete();
        }
        return response()->json([
            'message' => 'OK',
        ], Response::HTTP_NO_CONTENT);
    }

    public function cancel(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'purchase' => 'nullable|integer|exists:purchases,id',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors()->toArray(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        if ($request->filled('purchase')) {
            $purchase = Purchase::find($request->input('purchase'));
            abort_if(!request()->user()->can('update', $purchase), Response::HTTP_FORBIDDEN);
            $purchase->stockMoves()->forceDelete();
        } else {
            $company = request()->user()->company;
            $company->stockMoves()->doesntHave('purchase')->where('validated_at', null)->forceDelete();
        }

        return response()->json([
            'message' => 'OK',
        ], Response::HTTP_NO_CONTENT);
    }

    public function finalize(Request $request)
    {
        $company = request()->user()->company;
        $company->stockMoves()->where('validated_at', null)->get()->each(function ($item) {
            $item->validate();
        });
        return response()->json([
            'message' => 'OK',
        ], Response::HTTP_ACCEPTED);
    }
}
