<?php

namespace App\models\sales;

use Carbon\Carbon;
use App\InvoicerModel;
use App\models\users\User;
use App\models\shifts\Shift;
use App\models\company\Brunch;
use App\models\stock\Saleable;
use App\models\stock\StockMove;
use App\models\customers\Customer;
use App\models\stock\StockMoveType;
use App\models\customers\CustomerOperation;
use Staudenmeir\EloquentHasManyDeep\HasRelationships;

class SaleModel extends InvoicerModel
{
    use HasRelationships;

    protected $casts = [
        'validated_at'  => 'datetime',
        'due_date'      => 'datetime',
        'print_nr'      => 'integer'
    ];

    public function order()
    {
        $current = $this->print_nr;

        $this->print_nr = $this->print_nr + 1;
        $this->save();
        switch ($current) {
            case 0:
                return __('Original');
                break;
            case 1:
                return __('Duplicado');
                break;
            case 2:
                return __('Triplicado');
                break;
            default:
                return __('Cópia');
                break;
        }
    }


    public function taxAmount()
    {
        return $this->saleItens->sum(function ($item) {
            return $item->taxAmount();
        });
    }

    public function totalAmount()
    {
        return $this->saleItens->sum(function ($item) {
            return $item->totalAmount();
        });
    }

    public function vatResume()
    {
        return $this->saleItens->groupBy('vat_tax')->transform(function ($value, $key) {
            return (object) [
                'vat_tax'   => $key,
                'incidence' => $value->sum(function ($item) {
                    return ($item->totalAmount() - $item->taxAmount());
                }),
                'total'     =>  $value->sum(function ($item) {
                    return $item->taxAmount();
                }),
            ];
        });
    }

    public function groceriesAndServices()
    {
        return $this->saleItens->sum(function ($item) {
            return ($item->totalAmount() - $item->taxAmount());
        });
    }

    public function commercialDiscount()
    {
        return $this->saleItens->sum(function ($item) {
            return ($item->fullDiscount());
        });
    }



    public function hasSaleItem(Saleable $saleable)
    {
        return $this->saleItens()->whereHas('saleable', function ($query) use ($saleable) {
            $query->where(Saleable::table() . '.id', $saleable->id);
        })->first();
    }



    public function company()
    {
        return $this->hasOneDeepFromRelations($this->user(), (new User())->company());
    }

    public function user()
    {
        return $this->hasOneDeepFromRelations($this->shift(), (new Shift())->user());
    }
    public function shift()
    {
        return $this->belongsTo(Shift::class, 'shifts_id');
    }



    public function isCancelled()
    {
        return !is_null($this->anuled_at);
    }

    public function isValid()
    {
        return !is_null($this->validated_at);
    }

    public function canBeChanged(): bool
    {
        return !$this->isValid();
    }

    public function customerOperation()
    {
        return $this->belongsTo(CustomerOperation::class, 'customer_operations_id');
    }

    public function customer()
    {
        return $this->hasOneDeepFromRelations($this->customerOperation(), (new CustomerOperation())->customer());
    }

    private function fillCustomerAddress()
    {
        $this->customer_name = $this->customer->name;
        $this->customer_nuit = $this->customer->nuit;
        $this->customer_address = $this->customer->address;
        $this->save();
    }

    public function generateCustomerOperation(Customer $customer)
    {
        $customerOperation = $customer->createOperation();
        $customerOperation->validated_at = $this->validated_at;
        $customerOperation->save();
        $this->customerOperation()->associate($customerOperation);
        $this->save();
    }




    public function generateSequence()
    {
        $company = $this->company;
        $sale =  static::whereHas('shift', function ($query) use ($company) {
            $query->whereHas('company', function ($query) use ($company) {
                $query->where(Brunch::table() . '.id', $company->id);
            });
        })
            ->where(
                [
                    ['validated_at', '<>', null],
                    ['id', '<>', $this->id]
                ]
            )->orderBy('validated_at', 'DESC')->first();

        $this->document_sequence = $sale ? ($sale->document_sequence + 1) : 1;
        $this->save();
    }

    public function validate(Customer $customer = null)
    {
        $shift = $this->user->currentShift();
        $this->shift()->associate($shift);
        $this->validated_at = now();
        if ($customer) {
            $this->generateCustomerOperation($customer);
            $this->fillCustomerAddress();
            $this->applyCustomerDiscounts($customer);
        }

        $this->generateSequence();
        $this->moveStock();
        $this->save();
        return $this;
    }

    public function applyCustomerDiscounts(Customer $customer)
    {
        if (($customer->discount > 0 and $customer->discount < 100)) {;
            if ($this->financial_discount == 0 or is_null($this->financial_discount)) {
                $this->financial_discount =             getPercentageValue($this->totalAmount(), $customer->discount);
                $this->save();
            }
        }
    }

    public function  moveStock()
    {
        if ($this->isProforma() or $this->isReceipt())
            return;
        $itens = $this->saleItens()->whereHas('saleable', function ($query) {
            return $query->where('is_stockable', true);
        })->with(['saleable', 'warehouse']);

        $main = $this->company->saleWarehouse() ?? null;
        $moveType = StockMoveType::sale();
        $user = $this->user;

        if ($main)
            $itens->each(function ($item) use ($main, $moveType, $user) {
                $saleable = $item->saleable;
                $main = $item->warehouse ?? $main;


                $move = $main->lastMove($saleable);

                $qty =
                    $this->isCreditNote() ? ($item->quantity) : (0 - $item->quantity);
                if ($move) {
                    $fullPrice = $move->unit_price * $item->quantity;
                    $move = StockMove::move($saleable, $main, $moveType, $user, $qty, $fullPrice, $move->expirity_date);
                    $move->validate();
                }
            });
    }
    public function isCashSale()
    {
        return false;
    }

    public function isBasicSale()
    {
        return false;
    }

    public function isCreditNote()
    {
        return false;
    }

    public function isDebitNote()
    {
        return false;
    }

    public function isInvoice()
    {
        return false;
    }

    public function isProforma()
    {
        return false;
    }


    public function isReceipt()
    {
        return false;
    }
    public function isDeliveryNote()
    {
        return false;
    }



    public function issueDate()
    {
        return $this->validated_at->format('d/m/Y H:i');
    }

    public function dueDate()
    {
        return $this->due_date->format('d/m/Y');
    }

    public function operator()
    {
        return $this->user->name;
    }


    public function name()
    {
        return $this->id . '.' . (new Carbon($this->validated_at))->format("Y");
    }
}
