<?php

namespace App\models\stock;

use App\InvoicerModel;
use App\models\company\Brunch;
use App\models\stock\StockMove;
use App\models\provider\Purchase;
use App\models\customers\Customer;
use App\models\provider\ProviderOperation;
use Illuminate\Database\Eloquent\SoftDeletes;
use Staudenmeir\EloquentHasManyDeep\HasRelationships;

class Provider extends InvoicerModel
{
    use SoftDeletes;
    use HasRelationships;


    protected $table = 'providers';

    public function company()
    {
        return $this->belongsTo(Brunch::class, 'brunch_company_id');
    }

    public function stock_moves()
    {
        return $this->hasMany(StockMove::class, 'providers_id');
    }

    public function customer()
    {
        return $this->belongsTo(Customer::class, 'customers_id');
    }

    public function purchases()
    {
        return $this->hasManyDeepFromRelations($this->providerOperations(), (new ProviderOperation())->purchase());
    }

    public function expenses()
    {
        return $this->hasManyDeepFromRelations($this->providerOperations(), (new ProviderOperation())->expense());
    }

    public function payments()
    {
        return $this->hasManyDeepFromRelations($this->providerOperations(), (new ProviderOperation())->payment());
    }

    /**
     * @return [type]
     */
    public function providerOperations()
    {
        return $this->hasMany(ProviderOperation::class, 'providers_id');
    }

    public function pendingPurchases()
    {
        return $this->validPurchases->filter(function (Purchase $purchase) {
            return $purchase->pendingAmount() >= 0;
        });
    }

    public function totalPurchased()
    {
        return $this->validPurchases()->get()->sum(function (Purchase $purchase) {
            return $purchase->totalAmount();
        });
    }
    public function totalInDept()
    {
        return  $this->providerOperations()->where(function ($query) {
            return  $query->whereHas('expense', function ($query) {
            })
                ->orWhereHas('purchase', function ($query) {
                    return $query;
                });
        })->with(['expense', 'purchase'])->get()->sum(function ($operation) {
            return  $operation->expense ? $operation->expense->pendingAmount() : $operation->purchase->pendingAmount();
        });
    }

    public function balance()
    {
        $providerBalance =  $this->totalInDept();
        $customerBalance = 0;
        $customer = $this->customer;


        if ($customer)
            $customerBalance = $customer->balance();

        return $customerBalance + $providerBalance;
    }

    public function validPurchases()
    {
        return $this->purchases()->where([
            [Purchase::table() . '.validated_at', '<>', null],
            [Purchase::table() . '.anuled_at', null],
        ]);
    }

    public function unfinishedPurchase(): Purchase
    {
        $purchase = $this->purchases()->where(Purchase::table() . '.validated_at', null)->first();
        return $purchase ?? $this->createPurchase();
    }
    public function createPurchase()
    {
        $purchase = new Purchase();
        $purchase->save();
        return $purchase;
    }
}
